// content.js
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action !== "getTables") return;

  const extractTableName = (tableEl) => {
    const clean = (s) => (s || "").replace(/\s+/g, " ").trim();

    // 1) <caption>
    const caption = tableEl.querySelector("caption");
    if (caption && clean(caption.innerText)) return clean(caption.innerText);

    // 2) aria-label on table
    const ariaLabel = tableEl.getAttribute("aria-label");
    if (ariaLabel && clean(ariaLabel)) return clean(ariaLabel);

    // 3) Semrush specific: go up to the nearest "card" container
    const card = tableEl.closest(
      "[data-ui-name='Card'], [data-test-batch-table], section, article, [role='region']"
    );
    if (card) {
      // Prefer Semrush title block if exists
      const semrushTitle = card.querySelector(
        "[data-test-batch-table-title] h1, [data-test-batch-table-title] h2, [data-test-batch-table-title] h3"
      );
      if (semrushTitle && clean(semrushTitle.innerText))
        return clean(semrushTitle.innerText);

      // Any heading inside that card
      const heading = card.querySelector("h1, h2, h3, h4, h5, h6");
      if (heading && clean(heading.innerText)) return clean(heading.innerText);
    }

    // 4) Fallback: search upward and look at previous headings in ancestors
    let el = tableEl;
    for (let i = 0; i < 6 && el; i++) {
      let prev = el.previousElementSibling;
      while (prev) {
        const h =
          prev.querySelector?.("h1,h2,h3,h4,h5,h6") ||
          (/^H[1-6]$/.test(prev.tagName) ? prev : null);
        if (h && clean(h.innerText)) return clean(h.innerText);
        prev = prev.previousElementSibling;
      }
      el = el.parentElement;
    }

    return null;
  };

  const getTablesNow = () => {
    const tables = [];

    const htmlTables = Array.from(document.querySelectorAll("table"));
    if (htmlTables.length) {
      htmlTables.forEach((table) => {
        const rows = Array.from(table.rows).map((row) =>
          Array.from(row.cells).map((cell) => cell.innerText.trim())
        );

        if (rows.length) {
          tables.push({
            rows,
            name: extractTableName(table),
          });
        }
      });
      return tables;
    }

    // Grid-based tables (div role=row)
    const rowDivs = Array.from(document.querySelectorAll("div[role='row']"));
    if (rowDivs.length) {
      const rows = rowDivs.map((row) =>
        Array.from(row.querySelectorAll("div[role='gridcell']")).map((cell) =>
          cell.innerText.trim()
        )
      );

      if (rows.length) {
        tables.push({
          rows,
          name: null,
        });
      }
    }

    return tables;
  };

  let tries = 0;
  const maxTries = 12;

  const timer = setInterval(() => {
    const tables = getTablesNow();
    if (tables.length || tries >= maxTries) {
      clearInterval(timer);
      sendResponse({ tables });
    }
    tries++;
  }, 250);

  return true;
});
